"""
DAE: Diffusion‑Aligned Embeddings
=================================

This package contains a reference implementation of the Diffusion‑Aligned
Embeddings (DAE) algorithm.  DAE constructs a weighted k‑nearest neighbor graph
with self‑tuned bandwidths and optionally builds diffusion embeddings via a
continuous‑time Markov chain (CTMC) optimizer.

The functionality is organized into the :mod:`dae.graph_utils` subpackage,
which provides utilities for graph construction and symmetrization, and the
``ctmc_engine`` module which implements the embedding optimizer.

Example usage
-------------

Create a weighted k‑NN graph with self‑tuning sigmas::

    import numpy as np
    from dae.graph_utils import build_weighted_graph

    X = np.random.randn(100, 2).astype(np.float32)
    rho, sigmas, ei, ej, P_vals, neigh_idx = build_weighted_graph(X, k=15, symmetrize="max")

See the README for more details and advanced options.
"""

from . import graph_utils  # noqa: F401
# Attempt to import the CTMC engine.  It requires numba and other optional
# dependencies.  If the import fails (e.g. because numba is not installed)
# we simply skip it so that the graph_utils subpackage can still be used.
try:
    from . import ctmc_engine as _ctmc_engine  # noqa: F401
    from .ctmc_engine import *  # noqa: F401,F403
except Exception:
    # If optional dependencies are missing we silently ignore the CTMC module.
    pass

__all__ = []  # symbols are exported through submodules